<?php

if ( !defined( 'ABSPATH' ) ) {
    exit;
    // Exit if accessed directly
}


if ( !class_exists( 'ACFFrontend_Hooks' ) ) {
    class ACFFrontend_Hooks
    {
        public function acfef_date_time_field( $field )
        {
            if ( !isset( $field['custom_post_date'] ) ) {
                return $field;
            }
            $field['value'] = date( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), time() );
            return $field;
        }
        
        public function frontend_only_setting( $field )
        {
            acf_render_field_setting( $field, array(
                'label'        => __( 'Hidden Field' ),
                'instructions' => __( 'Lets you hide the field everywhere. Useful if you need hidden data', 'acf-frontend-form-element' ),
                'name'         => 'hidden_field',
                'type'         => 'true_false',
                'ui'           => 1,
            ), true );
            acf_render_field_setting( $field, array(
                'label'        => __( 'Show On Frontend Only' ),
                'instructions' => __( 'Lets you hide the field on the backend to avoid duplicate fields.', 'acf-frontend-form-element' ),
                'name'         => 'only_front',
                'type'         => 'true_false',
                'ui'           => 1,
                'conditions'   => [ [
                'field'    => 'hidden_field',
                'operator' => '==',
                'value'    => '0',
            ] ],
            ), true );
            acf_render_field_setting( $field, array(
                'label'        => __( 'Hide Field Label', 'acf-frontend-form-element' ),
                'instructions' => __( 'Lets you hide the field\'s label including HTML markup.', 'acf-frontend-form-element' ),
                'name'         => 'field_label_hide',
                'type'         => 'true_false',
                'ui'           => 1,
                'conditions'   => [ [
                'field'    => 'only_front',
                'operator' => '==',
                'value'    => '1',
            ] ],
            ), true );
        }
        
        public function read_only_setting( $field )
        {
            $types = array(
                'text',
                'textarea',
                'email',
                'number'
            );
            if ( in_array( $field['type'], $types ) ) {
                acf_render_field_setting( $field, array(
                    'label'        => __( 'Read Only', 'acf-frontend-form-element' ),
                    'instructions' => 'Prevent users from changing the data.',
                    'name'         => 'readonly',
                    'type'         => 'true_false',
                    'ui'           => 1,
                ) );
            }
        }
        
        public function hide_acfef_fields( $groups )
        {
            unset( $groups['acfef-hidden'] );
            return $groups;
        }
        
        private function acfef_is_custom( $field )
        {
            foreach ( $field as $key => $value ) {
                if ( 'custom_' == substr( $key, 0, 7 ) && $value == 1 ) {
                    return true;
                }
            }
            return false;
        }
        
        public function acfef_load_text_value( $value, $post_id = false, $field = false )
        {
            if ( !$this->acfef_is_custom( $field ) ) {
                return $value;
            }
            if ( $post_id ) {
                
                if ( strpos( $post_id, 'term_' ) !== false ) {
                    $term_id = explode( '_', $post_id )[1];
                    $edit_term = get_term( $term_id );
                    if ( !is_wp_error( $edit_term ) ) {
                        if ( isset( $field['custom_term_name'] ) && $field['custom_term_name'] == 1 ) {
                            $value = $edit_term->name;
                        }
                    }
                } elseif ( strpos( $post_id, 'comment' ) !== false ) {
                    $current_user = wp_get_current_user();
                    if ( $current_user !== 0 ) {
                        if ( isset( $field['custom_author'] ) && $field['custom_author'] == 1 ) {
                            $value = esc_html( $current_user->display_name );
                        }
                    }
                }
            
            }
            return $value;
        }
        
        public function acfef_load_email_value( $value, $post_id = false, $field = false )
        {
            if ( !$this->acfef_is_custom( $field ) ) {
                return $value;
            }
            if ( $post_id ) {
                
                if ( strpos( $post_id, 'comment' ) !== false ) {
                    $current_user = wp_get_current_user();
                    if ( $current_user !== 0 ) {
                        if ( isset( $field['custom_author_email'] ) && $field['custom_author_email'] == 1 ) {
                            $value = esc_html( $current_user->user_email );
                        }
                    }
                }
            
            }
            return $value;
        }
        
        public function acfef_load_user_value( $value, $post_id = false, $field = false )
        {
            if ( !$this->acfef_is_custom( $field ) ) {
                return $value;
            }
            if ( $post_id ) {
                
                if ( is_numeric( $post_id ) ) {
                    $edit_post = get_post( $post_id );
                    if ( isset( $field['custom_post_author'] ) && $field['custom_post_author'] == 1 ) {
                        $value = $edit_post->post_author;
                    }
                }
            
            }
            return $value;
        }
        
        public function acfef_load_date_time_value( $value, $post_id = false, $field = false )
        {
            if ( !$this->acfef_is_custom( $field ) ) {
                return $value;
            }
            if ( $post_id ) {
                
                if ( is_numeric( $post_id ) ) {
                    $edit_post = get_post( $post_id );
                    if ( isset( $field['custom_post_date'] ) && $field['custom_post_date'] == 1 ) {
                        $value = $edit_post->post_date;
                    }
                }
            
            }
            return $value;
        }
        
        public function update_acfef_values( $value, $post_id = false, $field = false )
        {
            if ( !empty($field['no_save']) ) {
                return null;
            }
            
            if ( isset( $_POST['_acf_status'] ) && $_POST['_acf_status'] == 'publish' ) {
                $revisions = wp_get_post_revisions( $post_id );
                
                if ( !empty($revisions[0]) ) {
                    remove_filter(
                        'acf/update_value',
                        [ $this, 'update_acfef_values' ],
                        7,
                        3
                    );
                    acf_update_value( $value, $revisions[0]->ID, $field );
                    add_filter(
                        'acf/update_value',
                        [ $this, 'update_acfef_values' ],
                        7,
                        3
                    );
                }
            
            }
            
            if ( $post_id !== 'acfef_options' ) {
                return $value;
            }
            update_option( $field['key'], $value );
            return null;
        }
        
        public function acfef_update_text_value( $value, $post_id = false, $field = false )
        {
            if ( !$this->acfef_is_custom( $field ) ) {
                return $value;
            }
            
            if ( strpos( $post_id, 'term' ) !== false ) {
                $term_id = explode( '_', $post_id )[1];
                $edit_term = get_term( $term_id );
                if ( !is_wp_error( $edit_term ) ) {
                    
                    if ( isset( $field['custom_term_name'] ) && $field['custom_term_name'] == 1 ) {
                        $update_args = array(
                            'name' => $value,
                        );
                        if ( $field['change_slug'] ) {
                            $update_args['slug'] = sanitize_title( $value );
                        }
                        wp_update_term( $term_id, $edit_term->taxonomy, $update_args );
                    }
                
                }
            } elseif ( strpos( $post_id, 'comment' ) !== false ) {
                $comment_id = explode( '_', $post_id )[1];
                $comment_to_edit = [
                    'comment_ID' => $comment_id,
                ];
                if ( isset( $field['custom_author'] ) && $field['custom_author'] == 1 ) {
                    $comment_to_edit['comment_author'] = esc_attr( $value );
                }
                wp_update_comment( $comment_to_edit );
            }
            
            return null;
        }
        
        public function acfef_update_email_value( $value, $post_id = false, $field = false )
        {
            if ( !$this->acfef_is_custom( $field ) ) {
                return $value;
            }
            
            if ( strpos( $post_id, 'comment' ) !== false ) {
                $comment_id = explode( '_', $post_id )[1];
                $comment_to_edit = [
                    'comment_ID' => $comment_id,
                ];
                if ( isset( $field['custom_author_email'] ) && $field['custom_author_email'] == 1 ) {
                    $comment_to_edit['comment_author_email'] = esc_attr( $value );
                }
                wp_update_comment( $comment_to_edit );
            }
            
            return null;
        }
        
        public function acfef_update_user_value( $value, $post_id = false, $field = false )
        {
            if ( !$this->acfef_is_custom( $field ) ) {
                return $value;
            }
            if ( isset( $field['custom_post_author'] ) && $field['custom_post_author'] == 1 ) {
                $_POST['post_fields']['post_author'] = $value;
            }
            return null;
        }
        
        public function acfef_update_date_time_value( $value, $post_id = false, $field = false )
        {
            if ( !$this->acfef_is_custom( $field ) ) {
                return $value;
            }
            
            if ( isset( $field['custom_post_date'] ) && $field['custom_post_date'] == 1 ) {
                $format = get_option( 'links_updated_date_format' );
                $_POST['post_fields']['post_date'] = $value;
            }
            
            return null;
        }
        
        public function acfef_after_save_post( $post_id )
        {
            if ( !isset( $_POST['acf'] ) ) {
                return $post_id;
            }
            $form = false;
            
            if ( isset( $_POST['_acf_form'] ) ) {
                // Load registered form using id.
                $form = acf()->form_front->get_form( $_POST['_acf_form'] );
                // Fallback to encrypted JSON.
                if ( !$form ) {
                    $form = json_decode( acf_decrypt( $_POST['_acf_form'] ), true );
                }
            }
            
            
            if ( is_numeric( $post_id ) ) {
                if ( isset( $_POST['_acf_element_id'] ) ) {
                    update_metadata(
                        'post',
                        $post_id,
                        'acfef_form_source',
                        $_POST['_acf_element_id']
                    );
                }
                
                if ( isset( $_POST['post_fields'] ) ) {
                    $post_to_edit = $_POST['post_fields'];
                    $post_to_edit['ID'] = $post_id;
                    remove_action(
                        'acf/save_post',
                        [ $this, 'acfef_after_save_post' ],
                        10,
                        1
                    );
                    wp_update_post( $post_to_edit );
                    add_action(
                        'acf/save_post',
                        [ $this, 'acfef_after_save_post' ],
                        10,
                        1
                    );
                }
            
            } elseif ( strpos( $post_id, 'user_' ) !== false ) {
                $user_id = explode( 'user_', $post_id )[1];
                $user_to_edit = get_user_by( 'ID', $user_id );
                if ( isset( $_POST['_acf_element_id'] ) ) {
                    update_user_meta( $user_id, 'acfef_form_source', $_POST['_acf_element_id'] );
                }
                if ( isset( $form['user_manager'] ) ) {
                    update_user_meta( $user_id, 'acfef_manager', $form['user_manager'] );
                }
                
                if ( isset( $form['display_name_default'] ) ) {
                    if ( $form['display_name_default'] == 'default' ) {
                        return;
                    }
                    switch ( $form['display_name_default'] ) {
                        case 'user_login':
                            $display_name = $user_to_edit->user_login;
                            break;
                        case 'user_email':
                            $display_name = $user_to_edit->user_email;
                            break;
                        case 'first_name':
                            $display_name = $user_to_edit->first_name;
                            break;
                        case 'last_name':
                            $display_name = $user_to_edit->last_name;
                            break;
                        case 'nickname':
                            $display_name = $user_to_edit->nickname;
                            break;
                        case 'first_last':
                            $display_name = $user_to_edit->first_name . ' ' . $user_to_edit->last_name;
                            break;
                    }
                    if ( isset( $display_name ) ) {
                        wp_update_user( [
                            'ID'           => $user_id,
                            'display_name' => $display_name,
                        ] );
                    }
                }
            
            }
        
        }
        
        public function acfef_exclude_groups( $field_group )
        {
            
            if ( empty($field_group['acfef_group']) ) {
                return $field_group;
            } elseif ( is_admin() ) {
                global  $acfef_settings ;
                
                if ( function_exists( 'get_current_screen' ) ) {
                    $current_screen = get_current_screen();
                    
                    if ( isset( $current_screen->post_type ) && $current_screen->post_type == 'acf_frontend_form' ) {
                        return $field_group;
                    } else {
                        return null;
                    }
                
                }
            
            }
        
        }
        
        public function acfef_validation()
        {
            if ( isset( $_POST['_acf_field_id'] ) ) {
                acf_add_local_field( array(
                    'key'    => 'acfef_post_type',
                    'label'  => __( 'Post Type', 'acf-frontend-form-element' ),
                    'name'   => 'acfef_post_type',
                    'type'   => 'post_type',
                    'layout' => 'vertical',
                ) );
            }
        }
        
        public function acfef_pass_validation()
        {
            if ( isset( $_POST['_acf_status'] ) && $_POST['_acf_status'] != 'publish' ) {
                acf_reset_validation_errors();
            }
        }
        
        public function acfef_validate_term_name(
            $is_valid,
            $value,
            $field,
            $input
        )
        {
            if ( !isset( $field['custom_term_name'] ) || $field['custom_term_name'] == 0 ) {
                return $is_valid;
            }
            if ( !isset( $_POST['_acf_taxonomy_type'] ) ) {
                return $is_valid;
            }
            
            if ( term_exists( $value, $_POST['_acf_taxonomy_type'] ) ) {
                $term_id = explode( 'term_', $_POST['_acf_post_id'] );
                
                if ( $term_id[1] ) {
                    $term_to_edit = get_term( $term_id[1] );
                    if ( $term_to_edit && $term_to_edit->name == $value ) {
                        return $is_valid;
                    }
                }
                
                return __( 'The term ' . $value . ' exists.', 'acf-frontend-form-element' );
            }
            
            return $is_valid;
        }
        
        public function acfef_fields_enqueue_scripts()
        {
            wp_enqueue_style( 'acfef' );
            wp_enqueue_style( 'acfef-modal' );
            wp_enqueue_script( 'acfef' );
            wp_enqueue_script( 'acfef-modal' );
            wp_enqueue_style( 'dashicons' );
        }
        
        public function prepare_field_hidden( $field )
        {
            if ( empty($field['hidden_field']) ) {
                return $field;
            }
            
            if ( isset( $field['wrapper']['class'] ) ) {
                $field['wrapper']['class'] .= ' acf-hidden';
            } else {
                $field['wrapper']['class'] = 'acf-hidden';
            }
            
            return $field;
        }
        
        public function prepare_field_frontend( $field )
        {
            // bail early if no 'admin_only' setting
            if ( empty($field['only_front']) ) {
                return $field;
            }
            $render = true;
            // return false if is admin (removes field)
            if ( is_admin() && !wp_doing_ajax() ) {
                $render = false;
            }
            if ( acf_frontend_edit_mode() ) {
                $render = true;
            }
            if ( !$render ) {
                return false;
            }
            // return\
            return $field;
        }
        
        public function prepare_field_column( $field )
        {
            if ( !empty($field['start_column']) ) {
                echo  '<div style="width:' . $field['start_column'] . '%" class="acf-column">' ;
            }
            if ( isset( $field['end_column'] ) ) {
                echo  '</div>' ;
            }
            // return\
            return $field;
        }
        
        public function include_field_types()
        {
            //general
            include_once 'fields/general/related-terms.php';
            include_once 'fields/general/submit-button.php';
            include_once 'fields/general/upload-image.php';
            include_once 'fields/general/upload-images.php';
            include_once 'fields/general/group.php';
            //include_once('fields/general/flexible-content.php');
            include_once 'fields/general/text.php';
            include_once 'fields/general/file.php';
            include_once 'fields/general/relationship.php';
            //post
            include_once 'fields/post/title.php';
            include_once 'fields/post/content.php';
            include_once 'fields/post/excerpt.php';
            include_once 'fields/post/slug.php';
            include_once 'fields/post/featured-image.php';
            include_once 'fields/post/post-type.php';
            include_once 'fields/post/menu-order.php';
            //user
            include_once 'fields/user/username.php';
            include_once 'fields/user/email.php';
            include_once 'fields/user/password.php';
            include_once 'fields/user/password-confirm.php';
            include_once 'fields/user/first-name.php';
            include_once 'fields/user/last-name.php';
            include_once 'fields/user/nickname.php';
            include_once 'fields/user/display-name.php';
            include_once 'fields/user/bio.php';
            include_once 'fields/user/role.php';
        }
        
        public function __construct()
        {
            add_action( 'acf/include_field_types', array( $this, 'include_field_types' ), 6 );
            add_action( 'acf/enqueue_scripts', [ $this, 'acfef_fields_enqueue_scripts' ] );
            //type=text
            add_filter(
                'acf/load_value/type=text',
                [ $this, 'acfef_load_text_value' ],
                10,
                3
            );
            add_filter(
                'acf/update_value/type=text',
                [ $this, 'acfef_update_text_value' ],
                9,
                3
            );
            add_filter(
                'acf/validate_value/type=text',
                [ $this, 'acfef_validate_term_name' ],
                10,
                4
            );
            add_filter( 'acf/prepare_field', array( $this, 'prepare_field_hidden' ), 3 );
            add_filter( 'acf/prepare_field', array( $this, 'prepare_field_frontend' ), 3 );
            add_filter( 'acf/prepare_field', array( $this, 'prepare_field_column' ), 3 );
            //type=date_time_picker
            add_filter( 'acf/prepare_field/type=date_time_picker', [ $this, 'acfef_date_time_field' ] );
            add_filter(
                'acf/update_value/type=date_time_picker',
                [ $this, 'acfef_update_date_time_value' ],
                9,
                3
            );
            //add_filter( 'acf/load_value/type=date_time_picker', [ $this, 'acfef_load_date_time_value'], 10, 3);
            //Add field settings by type
            add_action( 'acf/render_field_settings', [ $this, 'frontend_only_setting' ] );
            add_action( 'acf/render_field_settings', [ $this, 'read_only_setting' ] );
            //add_action( 'acf/render_field_settings/type=post_object',  [ $this, 'acfef_add_edit_field'] );
            add_filter( 'acf/get_field_types', [ $this, 'hide_acfef_fields' ] );
            add_filter(
                'acf/load_value/type=user',
                [ $this, 'acfef_load_user_value' ],
                10,
                3
            );
            add_filter(
                'acf/update_value/type=user',
                [ $this, 'acfef_update_user_value' ],
                9,
                3
            );
            add_action(
                'acf/save_post',
                [ $this, 'acfef_after_save_post' ],
                10,
                1
            );
            add_filter(
                'acf/update_value',
                [ $this, 'update_acfef_values' ],
                7,
                3
            );
            add_filter( 'acf/load_field_group', [ $this, 'acfef_exclude_groups' ] );
            add_action( 'acf/validate_save_post', [ $this, 'acfef_validation' ], 1 );
            add_action( 'acf/validate_save_post', [ $this, 'acfef_pass_validation' ], 999 );
            require_once __DIR__ . '/forms/frontend_forms.php';
            require_once __DIR__ . '/forms/classes/permissions.php';
            require_once __DIR__ . '/forms/helpers/data_fetch.php';
            require_once __DIR__ . '/forms/helpers/shortcodes.php';
            require_once __DIR__ . '/forms/helpers/permissions.php';
            require_once __DIR__ . '/forms/actions/action_base.php';
            //actions
            require_once __DIR__ . '/forms/actions/term.php';
            require_once __DIR__ . '/forms/actions/user.php';
            require_once __DIR__ . '/forms/actions/post.php';
            require_once __DIR__ . '/forms/actions/comment.php';
        }
    
    }
    acfef()->acf_extend = new ACFFrontend_Hooks();
}
