<?php
	
/**
 * Handles the admin part of the forms
 *
 * @since 1.0.0
 *
 */
class ACF_Frontend_Forms {

	
	/**
	 * Adds a form key to a form if one doesn't exist
	 * 
	 * @since 1.0.0
	 *
	 */
	function save_post( $post_id, $post ) {

		 // do not save if this is an auto save routine
		if( defined('DOING_AUTOSAVE') && DOING_AUTOSAVE ) {
			return $post_id;
		} 
		
		// bail early if not acfef form
		if( $post->post_type !== 'acf_frontend_form' ) {
			return $post_id;
		}

		if( empty( $post->post_name ) ) {
			
			$form_key = 'form_' . uniqid();

			remove_action( 'save_post', array( $this, 'save_post' ) );
			
			wp_update_post( array(
				'ID' => $post_id,
				'post_name' => $form_key,
			) );
			
			add_action( 'save_post', array( $this, 'save_post' ), 10, 2 );

		}else{
					// only save once! WordPress save's a revision as well.
			if( wp_is_post_revision($post_id) ) {
				return $post_id;
			}
			
			// verify nonce
			if( !acf_verify_nonce('frontend_form') ) {
				return $post_id;
			} 
			
			// disable filters to ensure ACF loads raw data from DB
			acf_disable_filters();
			
			// save fields
			if( !empty($_POST['acf_fields']) ) {
				
				// loop
				foreach( $_POST['acf_fields'] as $field ) {
					
					// vars
					$specific = false;
					$save = acf_extract_var( $field, 'save' );
					
					
					// only saved field if has changed
					if( $save == 'meta' ) {
						$specific = array(
							'menu_order',
							'post_parent',
						);
					}
					
					// set parent
					if( !$field['parent'] ) {
						$field['parent'] = $post_id;
					}
					
					// save field
					acf_update_field( $field, $specific );
					
				}
			}
			
			
			// delete fields
			if( !empty( $_POST['_acf_delete_fields'] ) ) {
				
				// clean
				$ids = explode('|', $_POST['_acf_delete_fields']);
				$ids = array_map( 'intval', $ids );
				
				
				// loop
				foreach( $ids as $id ) {
					
					// bai early if no id
					if( !$id ) continue;
					
					
					// delete
					acf_delete_field( $id );
					
				}
				
			}
			
			if( ! empty( $_POST['form'] ) ){
				$_POST['form']['ID'] = $post_id;
				$_POST['form']['title'] = $_POST['post_title'];
				$_POST['form']['key'] = $post->post_name;
				$this->update_form_post( $_POST['form'] );
			}
		
		}
				
	}

	public function update_form_post( $data = array() ) {
		
		
		// may have been posted. Remove slashes
		$data = wp_unslash( $data );
		
		
		// parse types (converts string '0' to int 0)
		$data = acf_parse_types( $data );
		
		
		// extract some args
		$extract = acf_extract_vars($data, array(
			'ID',
			'key',
			'title',
			'menu_order',
			'fields',
			'active',
			'_valid'
		));
		
		
		// vars
		$data = maybe_serialize( $data );		
		
		// save
		$save = array(
			'ID'			=> $extract['ID'],
			'post_status'	=> 'publish',
			'post_title'	=> $extract['title'],
			'post_name'		=> $extract['key'],
			'post_type'		=> 'acf_frontend_form',
			'post_excerpt'	=> sanitize_title($extract['title']),
			'post_content'	=> $data,
			'menu_order'	=> $extract['menu_order'],
		);
		
		// slash data
		// - WP expects all data to be slashed and will unslash it (fixes '\' character issues)
		$save = wp_slash( $save );
		
		
		// update the field group and update the ID
		if( ! empty( $data['ID'] ) ) {
			
			wp_update_post( $save );
			
		} else {
			
			$form_id = wp_insert_post( $save );
			
		}
		
		// return
		return $data;
		
	}
	
	
	/**
	 * Displays the form key after the title
	 *
	 * @since 1.0.0
	 *
	 */
	function edit_form_after_title() {
		
		global $post;

		if ( 'acf_frontend_form' == $post->post_type ){

			// render post data
			acf_form_data(array(
				'screen'		=> 'frontend_form',
				'post_id'		=> $post->ID,
				'delete_fields'	=> 0,
				'validation'	=> 0
			));

			if ( $post->post_name ) {			
				echo '<div id="edit-slug-box">';
				
				echo sprintf(  '%s: <code>[acf_frontend form="%s"]</code>', __( 'Shortcode', 'acf-frontend-form-element' ), $post->post_name );
			
				echo '</div>';
			}

		}

	}

	
	/**
	 * Adds custom columns to the listings page
	 *
	 * @since 1.0.0
	 *
	 */
	function manage_columns( $columns ) {
		
		$new_columns = array(
			'shortcode'		=> __( 'Shortcode', 'acf-frontend-form-element' ),
			//'fields' 	=> __( 'Fields', 'acf-frontend-form-element' ),
		);

		// Remove date column
		unset( $columns['date'] );
		
		return array_merge( array_splice( $columns, 0, 2 ), $new_columns, $columns );
		
	}
	
	
	/**
	 * Outputs the content for the custom columns
	 *
	 * @since 1.0.0
	 *
	 */
	function columns_content( $column, $post_id ) {
		
		//$form = acfef_get_form( $post_id );
		
		if ( 'shortcode' == $column ) {
			echo sprintf( '<code>[acf_frontend form="%s"]</code>', get_post_field( 'post_name', $post_id ) );
		} 
		
	}



	/**
	 * Hides the months filter on the forms listing page.
	 *
	 * @since 1.6.5
	 *
	 */
	function disable_months_dropdown( $disabled, $post_type ) {
        if ( 'acf_frontend_form' != $post_type ) {
        return $disabled;
        }

        return true;
    }


	/**
	 * Registers the form settings fields
	 *
	 * @since 1.0.0
	 *
	 */
	function mb_permissions() {
		global $form;

		
		$fields = array(		
			array(
				'key' => 'who_can_see',
				'label' => __( 'Who Can See This...', 'acf-frontend-form-element' ),
				'type' => 'select',
				'instructions' => '',
				'required' => 0,
				'choices' => array(
					'logged_in'  => __( 'Only Logged In Users', 'acf-frontend-form-element' ),
					'logged_out' => __( 'Only Logged Out', 'acf-frontend-form-element' ),
					'all'        => __( 'All Users', 'acf-frontend-form-element' ),
				),
			),
			array(
				'key' => 'by_role',
				'label' => __( 'Select By Role', 'acf-frontend-form-element' ),
				'type' => 'select',
				'instructions' => '',
				'conditional_logic' => array(
					array(
						array(
							'field' => 'who_can_see',
							'operator' => '==',
							'value' => 'logged_in',
						),
					),
				),
				'multiple' => 1,
				'ui' => 1,
				'choices' => acf_frontend_get_user_roles( array(), true ),
			),
			/* array(
				'key' => 'user_id',
				'label' => __( 'Select By User', 'acf-frontend-form-element' ),
				'type' => 'user',
				'instructions' => '',
				'conditional_logic' => array(
					array(
						array(
							'field' => 'who_can_see',
							'operator' => '==',
							'value' => 'logged_in',
						),
					),
				),
				'allow_null' => 0,
				'multiple' => 1,
				'ajax' => 1,
				'ui' => 1,
				'return_format' => 'id',
			), */
			array(
				'key' => 'dynamic',
				'label' => __( 'Dynamic Permissions', 'acf-frontend-form-element' ),
				'type' => 'select',
				'instructions' => '',
				'conditional_logic' => array(
					array(
						array(
							'field' => 'who_can_see',
							'operator' => '==',
							'value' => 'logged_in',
						),
					),
				),
				'choices' => acf_frontend_user_id_fields(),
				'allow_null' => 1,
			),
		);

		foreach( $fields as $field ){
			$field['wrapper'] = array(
				'width' => '',
				'class' => '',
				'id' => '',
			);
			$field['prefix'] = 'form';
			$field['name'] = $field['key'];
			$field['value'] = $form[$field['key']];
			acfef_render_field_wrap( $field );
		}

	}

	/*  mb_fields
	*
	*  This function will render the HTML for the medtabox 'Form Settings'
	*
	*  @type	function
	*  @date	28/09/13
	*  @since	5.0.0
	*
	*  @param	N/A
	*  @return	N/A
	*/
	function mb_actions() {
		global $form;
		$fields = array(		
			array(
				'key' => 'redirect',
				'label' => 'Redirect After Submit',
				'type' => 'select',
				'instructions' => '',
				'required' => 0,
				'wrapper' => array(
					'width' => '',
					'class' => '',
					'id' => '',
				),
				'choices' => array(
					'current' => 'Reload Current Page',
					'custom_url' => 'Custom URL',
					'referer' => 'Referer',
					'post_url' => 'Post URL',
				),
				'allow_null' => 0,
				'multiple' => 0,
				'ui' => 0,
				'return_format' => 'value',
				'ajax' => 0,
				'placeholder' => '',
			),
			array(
				'key' => 'custom_url',
				'label' => 'Custom Url',
				'type' => 'url',
				'instructions' => '',
				'required' => 0,
				'conditional_logic' => array(
					array(
						array(
							'field' => 'redirect',
							'operator' => '==',
							'value' => 'custom_url',
						),
					),
				),
				'placeholder' => '',
			),
			array(
				'key' => 'show_update_message',
				'label' => 'Success Message',
				'type' => 'true_false',
				'instructions' => '',
				'required' => 0,
				'conditional_logic' => 0,
				'message' => '',
				'ui' => 1,
				'ui_on_text' => '',
				'ui_off_text' => '',
			),
			array(
				'key' => 'update_message',
				'label' => '',
				'field_label_hide' => true,
				'type' => 'textarea',
				'instructions' => '',
				'required' => 0,
				'conditional_logic' => array(
					array(
						array(
							'field' => 'show_update_message',
							'operator' => '==',
							'value' => '1',
						),
					),
				),
				'placeholder' => '',
				'maxlength' => '',
				'rows' => '2',
				'new_lines' => '',
			),
		);
				
		foreach( $fields as $field ){
			$field['wrapper'] = array(
				'width' => '',
				'class' => '',
				'id' => '',
			);
			$field['prefix'] = 'form';
			$field['name'] = $field['key'];
			$field['value'] = $form[$field['key']];
			acfef_render_field_wrap( $field );
		}

	}
	/*  mb_settings
	*
	*  This function will render the HTML for the medtabox 'Form Settings'
	*
	*  @type	function
	*  @date	28/09/13
	*  @since	5.0.0
	*
	*  @param	N/A
	*  @return	N/A
	*/
	function mb_settings() {
		global $form;
		$fields = array(		
			array(
				'key' => 'main_action',
				'label' => 'Type',
				'type' => 'select',
				'instructions' => '',
				'required' => 0,
				'conditional_logic' => 0,
				'choices' => array(
					'edit_post' => 'Edit Post',
					'new_post' => 'New Post',
				),
				'default_value' => false,
				'allow_null' => 0,
				'multiple' => 0,
				'ui' => 0,
				'return_format' => 'value',
				'ajax' => 0,
				'placeholder' => '',
			),	
			array(
				'key' => 'post_to_edit',
				'label' => 'Post to Edit',
				'type' => 'select',
				'instructions' => '',
				'required' => 0,
				'conditional_logic' => array(
					array(
						array(
							'field' => 'main_action',
							'operator' => '==',
							'value' => 'edit_post',
						),
					),
				),
				'choices' => array(
					'current' => 'Current Post',
					'url_query' => 'URL Query',
					'select_post' => 'Select Post',
				),
				'default_value' => false,
				'allow_null' => 0,
				'multiple' => 0,
				'ui' => 0,
				'return_format' => 'value',
				'ajax' => 0,
				'placeholder' => '',
			),
			array(
				'key' => 'url_query_post',
				'label' => 'URL Query Key',
				'type' => 'text',
				'instructions' => '',
				'required' => 0,
				'conditional_logic' => array(
					array(
						array(
							'field' => 'main_action',
							'operator' => '==',
							'value' => 'edit_post',
						),
						array(
							'field' => 'post_to_edit',
							'operator' => '==',
							'value' => 'url_query',
						),
					),
				),
				'placeholder' => '',
			),
			array(
				'key' => 'select_post',
				'label' => 'Select Post',
				'name' => 'select_post',
				'prefix' => 'form',
				'value' => $form['select_post'],
				'type' => 'post_object',
				'instructions' => '',
				'required' => 0,
				'conditional_logic' => array(
					array(
						array(
							'field' => 'main_action',
							'operator' => '==',
							'value' => 'edit_post',
						),
						array(
							'field' => 'post_to_edit',
							'operator' => '==',
							'value' => 'select_post',
						),
					),
				),
				'post_type' => '',
				'taxonomy' => '',
				'allow_null' => 0,
				'multiple' => 0,
				'return_format' => 'object',
				'ui' => 1,
			),
		
		);
				
		foreach( $fields as $field ){
			$field['wrapper'] = array(
				'width' => '',
				'class' => '',
				'id' => '',
			);
			$field['prefix'] = 'form';
			$field['name'] = $field['key'];
			$field['value'] = $form[$field['key']];
			acfef_render_field_wrap( $field );
		}
		
	}

		/*
	*  mb_fields
	*
	*  This function will render the HTML for the medtabox 'acf-field-group-fields'
	*
	*  @type	function
	*  @date	28/09/13
	*  @since	5.0.0
	*
	*  @param	N/A
	*  @return	N/A
	*/
	
	function mb_fields() {
		
		// global
		global $post;
		$form_fields = array();

		$args = array(
			'post_type' => 'acf-field',
			'posts_per_page' => '-1',
			'post_parent' => $post->ID,
			'fields' => 'ids',
			'orderby' => 'menu_order', 
			'order' => 'ASC'
		);
		
		$fields_query = get_posts( $args );
		
		if ( $fields_query ) {
			foreach( $fields_query as $field ){
				$form_fields[] = acf_get_field( $field );
			}
		}

		
		// get fields
		$view = array(
			'fields'	=> $form_fields,
			'parent'	=> 0
		);
		
		// load view
		acf_get_view('field-group-fields', $view);
		
	}

	function admin_head() {
		
		// global
		global $post, $form;
		
		
		// set global var
		$form = $this->get_form_data( $post );
		
		// metaboxes
		 
		add_meta_box('acf-field-group-fields', __("Fields",'acf'), array($this, 'mb_fields'), 'acf_frontend_form', 'normal', 'high');		
		add_meta_box('acf-frontend-form-settings', __("Settings",'acf'), array($this, 'mb_settings'), 'acf_frontend_form', 'side', 'core');
		add_meta_box('acf-frontend-form-actions', __("Actions",'acf'), array($this, 'mb_actions'), 'acf_frontend_form', 'normal', 'core'); 
		add_meta_box('acf-frontend-form-permissions', __("Permissions",'acf'), array($this, 'mb_permissions'), 'acf_frontend_form', 'normal', 'core'); 
		
	}

	public function get_form_data( $post ){
		$form = maybe_unserialize( $post->post_content );
		
		if( ! $form ) $form = array();

		$form = acf_frontend_parse_args( $form, array(
			'redirect' => 'current',
			'custom_url' => '',
			'show_update_message' => 1,
			'update_message' => __( 'Your post has been updated successfully.', 'acf-frontend-form-element' ),
			'main_action' => 'edit_post',
			'post_to_edit' => 'current',
			'url_query_post' => '',
			'select_post' => '',
			'who_can_see' => '',
			'by_role' => '',
			'user_id' => '',
			'dynamic' => '',
		) );

		return $form;
	}

	function admin_enqueue_scripts() {
		
		// no autosave
		wp_dequeue_script('autosave');
		
		
		// custom scripts
		wp_enqueue_style('acf-field-group');
		wp_enqueue_script('acf-field-group');
		
		
		// localize text
		acf_localize_text(array(
			'The string "field_" may not be used at the start of a field name'	=> __('The string "field_" may not be used at the start of a field name', 'acf'),
			'This field cannot be moved until its changes have been saved'		=> __('This field cannot be moved until its changes have been saved', 'acf'),
			'Field group title is required'										=> __('Form title is required', 'acf'),
			'Move to trash. Are you sure?'										=> __('Move to trash. Are you sure?', 'acf'),
			'No toggle fields available'										=> __('No toggle fields available', 'acf'),
			'Move Custom Field'													=> __('Move Custom Field', 'acf'),
			'Checked'															=> __('Checked', 'acf'),
			'(no label)'														=> __('(no label)', 'acf'),
			'(this field)'														=> __('(this field)', 'acf'),
			'copy'																=> __('copy', 'acf'),
			'or'																=> __('or', 'acf'),
			'Null'																=> __('Null', 'acf'),
		));
		
		// localize data
		acf_localize_data(array(
		   	'fieldTypes' => acf_get_field_types_info()
	   	));
	   	
		// 3rd party hook
		do_action('acf/field_group/admin_enqueue_scripts'); 
		
	}

	function current_screen() {
		// validate screen
		if( !acf_is_screen('acf_frontend_form') ) return;


		// disable filters to ensure ACF loads raw data from DB
		acf_disable_filters();
		
		
		// enqueue scripts
		acf_enqueue_scripts();

		add_action('acf/input/admin_enqueue_scripts',		array($this, 'admin_enqueue_scripts'));
		add_action('acf/input/admin_head', 					array($this, 'admin_head'));
	}

	function ajax_query() {
		$fields = array(		
			array(
				'key' => 'select_post',
				'label' => 'Select Post',
				'name' => 'select_post',
				'prefix' => 'form',
				'type' => 'post_object',
				'instructions' => '',
				'required' => 0,
				'conditional_logic' => array(
					array(
						array(
							'field' => 'main_action',
							'operator' => '==',
							'value' => 'edit_post',
						),
						array(
							'field' => 'post_to_edit',
							'operator' => '==',
							'value' => 'select_post',
						),
					),
				),
				'post_type' => '',
				'taxonomy' => '',
				'allow_null' => 0,
				'multiple' => 0,
				'return_format' => 'object',
				'ui' => 1,
			),
		
		);
				
		foreach( $fields as $field ){
			$field['wrapper'] = array(
				'width' => '',
				'class' => '',
				'id' => '',
			);
			$field['prefix'] = 'form';
			$field['name'] = $field['key'];
			acf_add_local_field( $field );
		}
			
	}

    function __construct() {
        require_once( __DIR__ . '/post-types.php' );

		// Actions
		add_action( 'edit_form_after_title', array( $this, 'edit_form_after_title' ), 11, 0 );

		add_action( 'save_post', array( $this, 'save_post' ), 10, 2 );
		add_action( 'current_screen', array($this, 'current_screen'));
		
		add_filter( 'manage_acf_frontend_form_posts_columns', array( $this, 'manage_columns' ), 10, 1 );
		add_action( 'manage_acf_frontend_form_posts_custom_column', array( $this, 'columns_content' ), 10, 2 );
		add_filter( 'disable_months_dropdown', array( $this, 'disable_months_dropdown' ), 10, 2 );

		add_action('wp_ajax_acf/fields/post_object/query', array($this, 'ajax_query'), 4);
		add_action('wp_ajax_nopriv_acf/fields/post_object/query', array($this, 'ajax_query'), 4);
	}
	
}

new ACF_Frontend_Forms();
